Unit BJDrive;

Interface

Uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, FileCtrl;

Const
     BJMaxDrives = 26;

Type
  TDriveFilter = Set Of (dfRemovable, dfFixed, dfCDROM, dfNetwork, dfRAMDisk);

  TBJDrive = Class(TCustomComboBox)
  Private
     { Private-Deklarationen }
     Drives: Array[0..BJMaxDrives-1] Of Char;
     DriveNum: Integer;
     PDrive: Char;
     PUserOnChange: Boolean;
     PDirList: TDirectoryListBox;
     fDriveFilter: TDriveFilter;
     fUseDriveFilter: Boolean;
     Procedure   SetUseDriveFilter(Value: Boolean);
     Procedure   SetDriveFilter(Value: TDriveFilter);
     Function    GetDrive: Char;
     Procedure   SetDrive(d: Char);
     Procedure   ResetItemHeight;
     Procedure   SetDirList(DirListBox: TDirectoryListBox);
     Procedure   CMFontChanged(Var Message: TMessage); Message CM_FONTCHANGED;
  Protected
     { Protected-Deklarationen }
     FloppyBMP, FixedBMP, NetworkBMP, CDROMBMP, RAMBMP: TBitmap;
     Procedure   Click; Override;
     Procedure   CreateWnd; Override;
     Procedure   DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); Override;
  Public
     { Public-Deklarationen }
     Constructor Create(AOwner: TComponent); Override;
     Destructor  Destroy; Override;
     procedure   UpdateDriveList;
     Property    Drive: Char Read GetDrive Write SetDrive;
  Published
     { Published-Deklarationen }
     Property Color;
     Property Ctl3D;
     Property DragMode;
     Property DragCursor;
     Property DriveFilter: TDriveFilter Read fDriveFilter Write SetDriveFilter;
     Property Enabled;
     Property Font;
     Property ImeMode;
     Property ImeName;
     Property ParentColor;
     Property ParentCtl3D;
     Property ParentFont;
     Property ParentShowHint;
     Property PopupMenu;
     Property ShowHint;
     Property TabOrder;
     Property TabStop;
     Property Visible;
     Property UseDriveFilter: Boolean Read fUseDriveFilter Write SetUseDriveFilter;
     Property UserOnChange: Boolean Read PUserOnChange Write PUserOnChange;
     Property DirList: TDirectoryListBox Read PDirList Write SetDirList;
     Property OnChange;
     Property OnClick;
     Property OnDblClick;
     Property OnDragDrop;
     Property OnDragOver;
     Property OnDropDown;
     Property OnEndDrag;
     Property OnEnter;
     Property OnExit;
     Property OnKeyDown;
     Property OnKeyPress;
     Property OnKeyUp;
     Property OnStartDrag;
  End;

Procedure Register;

Implementation

{$R icons.res}

Constructor TBJDrive.Create;
Var
   i: Integer;
Begin
     Inherited Create(AOwner);

     FloppyBMP := TBitmap.Create;
     FloppyBMP.Handle := LoadBitmap(HInstance, 'BJDISK');
     FixedBMP := TBitmap.Create;
     FixedBMP.Handle := LoadBitmap(HInstance, 'BJFIXED');
     NetworkBMP := TBitmap.Create;
     NetworkBMP.Handle := LoadBitmap(HInstance, 'BJNETWORK');
     CDROMBMP := TBitmap.Create;
     CDROMBMP.Handle := LoadBitmap(HInstance, 'BJCDROM');
     RAMBMP := TBitmap.Create;
     RAMBMP.Handle := LoadBitmap(HInstance, 'BJRAM');

     Style:=csOwnerDrawFixed;
     For i:=0 To BJMaxDrives-1 Do Drives[i]:=#0;
     DriveNum:=0;
     UserOnChange:=True;
     PDirList:=Nil;
     ResetItemHeight;

     fDriveFilter:=[];
     fUseDriveFilter:=False;
End;

Destructor TBJDrive.Destroy;
Begin
     FloppyBMP.Free;
     FixedBMP.Free;
     NetworkBMP.Free;
     CDROMBMP.Free;
     RAMBMP.Free;
     Inherited;
End;

Procedure TBJDrive.SetDriveFilter;
Var
   old: TDriveFilter;
Begin
     old:=fDriveFilter;
     fDriveFilter:=Value;
     If old <> fDriveFilter Then UpdateDriveList;
End;

Procedure TBJDrive.SetUseDriveFilter;
Var
   old: Boolean;
Begin
     old:=fUseDriveFilter;
     fUseDriveFilter:=Value;
     If old <> fUseDriveFilter Then UpdateDriveList;
End;

Procedure TBJDrive.DrawItem;
Var
  Bitmap: TBitmap;
  bmpWidth: Integer;
Begin
  with Canvas do
  begin
    FillRect(Rect);
    bmpWidth  := 16;
    Bitmap := TBitmap(Items.Objects[Index]);
    if Bitmap <> nil then
    begin
      bmpWidth := Bitmap.Width;
      BrushCopy(Bounds(Rect.Left + 2,
               (Rect.Top + Rect.Bottom - Bitmap.Height) div 2,
               Bitmap.Width, Bitmap.Height),
               Bitmap, Bounds(0, 0, Bitmap.Width, Bitmap.Height),
               Bitmap.Canvas.Pixels[0, Bitmap.Height - 1]);
    end;
     { uses DrawText instead of TextOut in order to get clipping against
       the combo box button   }
    Rect.Left := Rect.Left + bmpWidth + 6;
    DrawText(Canvas.Handle, PChar(Items[Index]), -1, Rect,
             DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX);
  end;
end;

Procedure TBJDrive.UpdateDriveList;
Var
   i: Integer;
   DriveBits: Longint;
   Dummy: Longint;
   Counter: Longint;
   NotUsed, VolFlags: Cardinal;
   Buf: array [0..MAX_PATH] of Char;
   VolumeLabel: String;
   DriveType: Longint;
   OldDrive: Char;
   bmp: TBitmap;
   dontAdd: Boolean;
Begin
     OldDrive:=Drive;
     For i:=0 To BJMaxDrives-1 Do Drives[i]:=#0;
     DriveNum:=0;
     DriveBits:=GetLogicalDrives;
     Dummy:=1;
     Items.Clear;
     Counter:=0;
     While Counter < BJMaxDrives Do Begin
           If (Dummy And DriveBits) = Dummy Then Begin
              Drives[DriveNum]:=Chr(Ord('A')+Counter);
              DriveType:=GetDriveType(PChar(Drives[DriveNum]+':\'));

              dontAdd:=False;
              If UseDriveFilter Then Begin
                 If (DriveType = DRIVE_REMOVABLE) And (Not (dfRemovable In DriveFilter)) Then dontAdd:=True;
                 If (DriveType = DRIVE_FIXED) And (Not (dfFixed In DriveFilter)) Then dontAdd:=True;
                 If (DriveType = DRIVE_CDROM) And (Not (dfCDROM In DriveFilter)) Then dontAdd:=True;
                 If (DriveType = DRIVE_REMOTE) And (Not (dfNetwork In DriveFilter)) Then dontAdd:=True;
                 If (DriveType = DRIVE_RAMDISK) And (Not (dfRAMDisk In DriveFilter)) Then dontAdd:=True;
              End;

              If (Not dontAdd) Then Begin
                 Case DriveType Of
                      DRIVE_FIXED, DRIVE_CDROM:
                         Begin
                              VolumeLabel:='';
                              If GetVolumeInformation(PChar(Drives[DriveNum]+':\'), Buf, SizeOf(Buf),
                                 Nil, NotUsed, VolFlags, Nil, 0) Then
                                      SetString(VolumeLabel, Buf, StrLen(Buf));
                              If VolumeLabel <> '' Then
                                 VolumeLabel:='['+AnsiLowerCase(VolumeLabel)+']'
                              Else Begin
                                 If DriveType = DRIVE_FIXED Then
                                    VolumeLabel:='Festplatte'
                                 Else
                                    VolumeLabel:='CD-DROM';
                              End;
                         End;
                      DRIVE_REMOVABLE:
                         VolumeLabel:='Disk';
                      DRIVE_REMOTE:
                         VolumeLabel:='Netzlaufwerk';
                      DRIVE_RAMDISK:
                         VolumeLabel:='RAM-Disk';
                 End;

                 Case DriveType Of
                      DRIVE_REMOVABLE:
                         bmp:=FloppyBMP;
                      DRIVE_FIXED:
                         bmp:=FixedBMP;
                      DRIVE_CDROM:
                         bmp:=CDROMBMP;
                      DRIVE_REMOTE:
                         bmp:=NetworkBMP;
                      DRIVE_RAMDISK:
                         bmp:=RamBMP;
                 Else
                      bmp:=Nil;
                 End;

                 Items.AddObject(Drives[DriveNum]+': '+VolumeLabel, TOBject(bmp));
                 Inc(DriveNum);
              End;
           End;
           Dummy:=Dummy Shl 1;
           Inc(Counter);
     End;
     Drive:=OldDrive;
End;

Procedure TBJDrive.CreateWnd;
Var
   Tmp: ShortString;
Begin
     Inherited CreateWnd;
     Drive:=#0;
     UpdateDriveList;
     GetDir(0,Tmp);
     If Tmp <> '' Then Drive:=Tmp[1];
End;

Function TBJDrive.GetDrive;
Begin
     Result:=PDrive;
End;

Procedure UpdateDirList(DirList: TDirectoryListBox; d: Char);
Begin
     If Assigned(DirList) Then DirList.Directory:=d+':';
End;

Procedure TBJDrive.SetDrive;
Var
   i: Integer;
Begin
     If d = #0 Then
        PDrive:=#0
     Else If DriveNum > 0 Then Begin
        For i:=0 To DriveNum-1 Do Begin
            If UpCase(d) = UpCase(Drives[i]) Then Begin
               PDrive:=Drives[i];
               ItemIndex:=i;
               If UserOnChange Then Begin
                  UpdateDirList(PDirList,PDrive);
                  Change;
               End;
               Exit;
            End;
        End;
     End;
End;

Procedure TBJDrive.Click;
Begin
     Inherited Click;
     If (ItemIndex >= 0) And (ItemIndex < DriveNum) Then Begin
        PDrive:=Drives[ItemIndex];
        UpdateDirList(PDirList,PDrive);
     End;
End;

Function GetItemHeight(Font: TFont): Integer;
Var
  DC: HDC;
  SaveFont: HFont;
  Metrics: TTextMetric;
Begin
     DC := GetDC(0);
     SaveFont := SelectObject(DC, Font.Handle);
     GetTextMetrics(DC, Metrics);
     SelectObject(DC, SaveFont);
     ReleaseDC(0, DC);
     Result := Metrics.tmHeight;
End;

Procedure TBJDrive.ResetItemHeight;
Var
   nuHeight: Integer;
Begin
     nuHeight :=  GetItemHeight(Font);
     If nuHeight > ItemHeight Then ItemHeight := nuHeight;
End;

Procedure TBJDrive.CMFontChanged(Var Message: TMessage);
Begin
     Inherited;
     ResetItemHeight;
     RecreateWnd;
End;

Procedure TBJDrive.SetDirList;
Begin
     PDirList:=DirListBox;
End;

Procedure Register;
Begin
     RegisterComponents('Beispiele', [TBJDrive]);
End;

End.
